/*
    Copyright (C) 2008  Johannes Bretscher

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "mainwin.h"
#include <QGraphicsScene>
#include <QFileDialog>
#include <QMessageBox>
#include <QGraphicsPixmapItem>
#include <QList>
#include <QCompleter>
#include <QDir>
#include <QSettings>
#include <iostream>

/*!
  Constructor
*/
MainWin::MainWin(QWidget *p)
	: QMainWindow(p) {
  m_ui.setupUi(this);

  m_tagsuche = NULL;

  m_bild = new QGraphicsScene();
  m_ui.bild->setScene(m_bild);

  m_bildauswahl = new Bildauswahl(this);
  addToolBar(Qt::TopToolBarArea, m_bildauswahl);

  QSettings settings;

  QObject::connect(m_ui.actionNeu, SIGNAL(triggered()),
	this, SLOT(newPicture()));
  QObject::connect(m_ui.actionSpeichern, SIGNAL(triggered()),
	this, SLOT(saveData()));
  QObject::connect(m_ui.actionSuchen, SIGNAL(triggered()),
	this, SLOT(search()));
  QObject::connect(m_ui.actionOptionen, SIGNAL(triggered()),
	this, SLOT(options()));
  QObject::connect(m_ui.ZoomIn, SIGNAL(clicked()),
	this, SLOT(zoomIn()));
  QObject::connect(m_ui.ZoomOut, SIGNAL(clicked()),
	this, SLOT(zoomOut()));
  QObject::connect(m_ui.tagInput, SIGNAL(returnPressed()),
	this, SLOT(newTag()));
  QObject::connect(m_ui.tagSelect, SIGNAL(itemSelectionChanged ()),
	this, SLOT(setDirty()));
  QObject::connect(m_bildauswahl, SIGNAL(bildGeaendert(const QString&)),
	this, SLOT(displayPicture(const QString&)));

  // Can this be shortened?
  QString dbPath = settings.value("db/path", "*FALSE*").toString();
  if("*FALSE*" == dbPath) {
    QDir tmpPath(QDir::homePath()+"/.imageTagger");
    if(!tmpPath.exists()) {
      tmpPath.mkpath(tmpPath.absolutePath());
    }
    dbPath = tmpPath.absolutePath();
    settings.setValue("db/path", dbPath);
  }
  m_datenbank = new Datenbank(dbPath);
  m_ui.tagSelect->addItems(m_datenbank->getAllTags());
  m_ui.tagInput->setCompleter(new QCompleter(m_ui.tagSelect->model()));


  m_path = QDir::currentPath();
  m_dirty = false;
}

/*!
  \~german Lade neue Bilder zum Betrachten und taggen
*/
void MainWin::newPicture() {
  newPictureList(QFileDialog::getOpenFileNames(this, tr("Bild oeffnen"), m_path,
	tr("Images (*.png *.jpg *.jpeg *.JPG *.xpm *.gif)")));
}

/*!
  \~german
  Neues Bild ausgewaehlt -> Anzeigen und Tags markieren
  \param bildName Pfad zum neuen Bild
*/
void MainWin::displayPicture(const QString &bildName) {
  if(m_dirty) {
    if(QMessageBox::Yes == QMessageBox::question(this,
    	tr("Tags wurden geaendert"),
	tr("Speichern?"),
	QMessageBox::Yes|QMessageBox::No,
	QMessageBox::Yes)) {
      saveData();
    }	
  }

  QList<QGraphicsItem *> bilder = m_bild->items();
  for(QList<QGraphicsItem *>::iterator bild = bilder.begin();
	bilder.end() != bild;
	bild++) {
    m_bild->removeItem(*bild);
  }
  
  m_bildname = bildName;

  if(!m_bildname.isEmpty()) {
    QImage showBild(m_bildname);
    if(showBild.isNull()) {
      QMessageBox::warning(this, tr("Fehler"),
	tr("Bild %1 konnte nicht geladen werden").arg(m_bildname) );
    } else {
      m_ui.bild->resetMatrix();
      m_bild->addItem(new QGraphicsPixmapItem(QPixmap::fromImage(showBild)));
      m_ui.bild->setSceneRect(m_bild->itemsBoundingRect());
    }
  } else {
    m_ui.bild->setSceneRect(m_bild->itemsBoundingRect());
  }

  statusBar()->showMessage(m_bildname);

  m_path = m_bildname.left(m_bildname.lastIndexOf('/'));

  // Tagdisplay aktualisieren
  QList<QListWidgetItem*> selectedTags = m_ui.tagSelect->selectedItems();
  QList<QListWidgetItem*>::iterator itTag;
  for(itTag = selectedTags.begin();
  	selectedTags.end() != itTag;
	itTag++) {
    (*itTag)->setSelected(false);
  }
  QStringList taglist = m_datenbank->getTags(m_bildname);
  QStringList::iterator itNewtag;
  for(itNewtag=taglist.begin();
	taglist.end() != itNewtag;
	itNewtag++) {
    m_ui.tagSelect->findItems(*itNewtag, Qt::MatchExactly).first()->setSelected(true);
  }
  m_dirty = false;
}

/*!
  \~german
  Tags zum Bild in die Datenbank schreiben
  \~english
  Write tags into database
*/
void MainWin::saveData() {
  QStringList taglist;
  QList<QListWidgetItem*> selectedTags = m_ui.tagSelect->selectedItems();
  QList<QListWidgetItem*>::iterator itTag;
  for(itTag = selectedTags.begin();
  	selectedTags.end() != itTag;
	itTag++) {
    taglist.append((*itTag)->text());
  }
  m_datenbank->writeTags(m_bildname, taglist);
  m_dirty = false;
}

/*!
  \~german
  Bildanzeige vergroessern
  \~english
  Increase picture display
*/
void MainWin::zoomIn() {
  m_ui.bild->scale(2.0, 2.0);
}

/*!
  \~german
  Bildanzeige verkleinern
  \~english
  Decrease picture display
*/
void MainWin::zoomOut() {
  m_ui.bild->scale(0.5, 0.5);
}

/*!
  \~german
  Tag wurde in der Eingabezeile eingegeben.
  \~english
  New tags entered
*/
void MainWin::newTag() {
  QString tagText = m_ui.tagInput->text();
  QList<QListWidgetItem*> tagList = m_ui.tagSelect->findItems(tagText, Qt::MatchExactly);
  if(!tagList.empty()) {
    tagList.first()->setSelected(true);
  } else {
    QListWidgetItem *newTagItem = new QListWidgetItem(tagText, m_ui.tagSelect);
    newTagItem->setSelected(true);
  }
  m_ui.tagInput->clear();

  setDirty();
}

/*!
  \~german
  Tags wurden geaendert
  \~english
  Tags have been changed
*/
void MainWin::setDirty() {
  m_dirty = true;
}

/*!
  Suche Bilder nach Tags
*/
void MainWin::search() {
  m_tagsuche = new Tagsuche(m_datenbank, this);
  QObject::connect(m_tagsuche, SIGNAL(selected(const QStringList &)),
	this, SLOT(newPictureList(const QStringList &)));
  m_tagsuche->show();
}

void MainWin::options() {
  Optionen optionen(this);
  optionen.exec();

}
/*!
  Durch Laden oder Bildsuche hat sich die Liste der Bilder geaendert.
*/
void MainWin::newPictureList(const QStringList &list) {
  // std::cerr << "Piclist" << std::endl;
  m_bildauswahl->setImageList(list);
}
