/*!
 \author Johannes Bretscher <bretscher@5sl.org>
 \date 2008-05-20
*/
/*
    Copyright (C) 2008  Johannes Bretscher

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "datenbank.h"
#include <iostream>
#include <QSqlIndex>
#include <QSqlQuery>
#include <QVariant>
#include <QSqlError>

const QString CONNECTION_NAME = "imageTagger_SQLITE_database_connection";

/*!
  Constructor
  \param db_path Pfad zur Datenbank
*/
Datenbank::Datenbank(const QString &db_path) {
  m_db = QSqlDatabase::addDatabase("QSQLITE", CONNECTION_NAME);
  m_db.setHostName("localhost");
  m_db.setDatabaseName(db_path+"/imageTagger.db");
  if(!m_db.open()) {
    std::cerr << "Kann Datenbank nicht öffnen" << std::endl;
  }
  if(m_db.primaryIndex("bild").isEmpty()) {
    createDatabase();
  }
}

Datenbank::~Datenbank() {
  m_db.close();
}

/*!
  Datenbank nicht gefunden, also neu anlegen.
*/
void Datenbank::createDatabase() {
  QString query;
  query = "CREATE TABLE bild ("
  	"bildid INTEGER PRIMARY KEY,"
	"name TEXT UNIQUE ON CONFLICT IGNORE)";
  QSqlQuery(query, m_db).exec();	
  query = "CREATE TABLE tags ("
	"bildid INTEGER NOT NULL,"
	"tag TEXT NOT NULL)";
  QSqlQuery(query, m_db).exec();	
  query = "CREATE INDEX idxBildName "
  	"ON bild (name)";
  QSqlQuery(query, m_db).exec();	
  query = "CREATE INDEX idxTags "
  	"ON tags (tag)";
  QSqlQuery(query, m_db).exec();	
}

/*!
  Alle zu einem Bild gehoerenden Tags heraussuchen.
  \param bild Vollstaendiger Name des Bildes
  \return Liste der zugehoerigen Tags. Leer, wenn Bild nicht in Datenbank.
*/
QStringList Datenbank::getTags(const QString &bild) const {
  QStringList ret;
  QSqlQuery qry(m_db);
  
  qry.prepare("SELECT tags.tag "
  	"FROM tags,bild "
	"WHERE tags.bildid=bild.bildid "
		"AND bild.name=?");
  qry.bindValue(0, bild);
  if(!qry.exec())
    std::cerr << "Datenbankfehler: \n"
        << "Query: " << qry.lastQuery().toUtf8().constData() << '\n'
    	<< qry.lastError().databaseText().toUtf8().constData() << '\n'
	<< qry.lastError().driverText().toUtf8().constData() << std::endl;
  while(qry.next()) {
    ret.append(qry.value(0).toString());
  }
  return ret;
}

/*!
  Schreibt die zum Bild gehoerigen Tags in die Datenbank

  Eventuelle bereits vorhandene Tags, die jetzt nicht mehr in der Liste sind
  werden geloescht.

  \param bild Vollstaendiger Name des Bildes
  \param taglist Vollstaendige Liste der Tags.
*/
void Datenbank::writeTags(const QString &bild, const QStringList &taglist) const {
  QSqlQuery qry(m_db);

  qry.prepare("INSERT INTO bild "
	"(name) "
	"VALUES (?)");
  qry.bindValue(0, bild);
  if(!qry.exec())
    std::cerr << "Datenbankfehler: \n"
        << "Query: " << qry.lastQuery().toUtf8().constData() << '\n'
    	<< qry.lastError().databaseText().toUtf8().constData() << '\n'
	<< qry.lastError().driverText().toUtf8().constData() << std::endl;

  qry.prepare("SELECT bild.bildid "
  	"FROM bild "
	"WHERE bild.name=?");
  qry.bindValue(0, bild);
  if(!qry.exec())
    std::cerr << "Datenbankfehler: \n"
        << "Query: " << qry.lastQuery().toUtf8().constData() << '\n'
    	<< qry.lastError().databaseText().toUtf8().constData() << '\n'
	<< qry.lastError().driverText().toUtf8().constData() << std::endl;
  qry.first();
  int uid = qry.value(0).toInt();

  qry.prepare("DELETE FROM tags "
  	"WHERE tags.bildid=?");
  qry.bindValue(0, uid);
  if(!qry.exec())
    std::cerr << "Datenbankfehler: \n"
        << "Query: " << qry.lastQuery().toUtf8().constData() << '\n'
    	<< qry.lastError().databaseText().toUtf8().constData() << '\n'
	<< qry.lastError().driverText().toUtf8().constData() << std::endl;

  QStringList::const_iterator it;
  for(it = taglist.begin();
  	taglist.end() != it;
	it++) {
    qry.prepare("INSERT INTO tags "
	"(bildid, tag) "
	"VALUES (?, ?)");
    qry.bindValue(0, uid);
    qry.bindValue(1, *it);
    if(!qry.exec())
      std::cerr << "Datenbankfehler: \n"
        << "Query: " << qry.lastQuery().toUtf8().constData() << '\n'
    	<< qry.lastError().databaseText().toUtf8().constData() << '\n'
	<< qry.lastError().driverText().toUtf8().constData() << std::endl;
  }
}

/*!
  Sucht alle Bilder die mit einem Tag versehen sind.

  \param tag Der Tag, nach dem gesucht wird.
  \return Liste der gefundenen Bilder. Leer bei nicht vergebenem Tag.
*/
QStringList Datenbank::findByTag(const QString &tag) const {
  QStringList ret;
  QSqlQuery qry(m_db);
  
  qry.prepare("SELECT bild.name "
  	"FROM tags,bild "
	"WHERE tags.bildid=bild.bildid "
		"AND tags.tag=?");
  qry.bindValue(0, tag);
  if(!qry.exec())
    std::cerr << "Datenbankfehler: \n"
        << "Query: " << qry.lastQuery().toUtf8().constData() << '\n'
    	<< qry.lastError().databaseText().toUtf8().constData() << '\n'
	<< qry.lastError().driverText().toUtf8().constData() << std::endl;
  while(qry.next()) {
    ret.append(qry.value(0).toString());
  }
  return ret;
}

/*!
  Sucht Bilder ueber erweiterte Suche 

  \param mustHave: Liste der Tags, die das Bild haben muss.
  \param notHave: Liste der Tags, die das Bild nicht haben darf.
  \return Liste der gefundenen Bilder. Leer bei nicht vergebenem Tag.
*/
QStringList Datenbank::findByTagList(const QStringList &mustHave, const QStringList& notHave) const {
  QStringList ret;
  QSqlQuery qry(m_db);
  int i;

  if(0 == mustHave.size() && 0 == notHave.size())
    return getAllImages();	// keine Einschraenkungen

  QString queryString = "SELECT bild.name FROM bild",
  	tempString = "";
  for(i=0; i<mustHave.size(); i++) {
    queryString += ",tags AS T" + QString::number(i);
    if(0 != i)
      tempString += " AND";
    tempString += " T" + QString::number(i) + ".tag=? AND T" +
                  QString::number(i) + ".bildid=bild.bildid";
  }
  
  queryString += " WHERE" + tempString;

  if(0 != notHave.size()) {
    if(0 != mustHave.size()) {
      queryString += " AND";
    }  
    queryString += " bild.bildid NOT IN"
    		   " ( SELECT DISTINCT bildid"
		     " FROM tags WHERE"
		     " tag IN"
		     " ( "; 
    		     
    for(i=0; i<notHave.size(); i++) {
      if(0 != i)
	queryString += ',';
      queryString += '?';
    }
    queryString +=   " )"
		   ")";
  }
  qry.prepare(queryString);
  int ct;
  for(i=0, ct=0; i<mustHave.size(); i++, ct++) {
    qry.bindValue(ct, mustHave[ct]);
    //std::cerr << ct << '\t' << mustHave[ct].toUtf8().constData() << std::endl;
  }
  for(i=0; i<notHave.size(); i++, ct++) {
    qry.bindValue(ct, notHave[i]);
    //std::cerr << ct << '\t' << notHave[i].toUtf8().constData() << std::endl;
  }
  if(!qry.exec())
    std::cerr << "Datenbankfehler: \n"
        << "Query: " << qry.lastQuery().toUtf8().constData() << '\n'
    	<< qry.lastError().databaseText().toUtf8().constData() << '\n'
	<< qry.lastError().driverText().toUtf8().constData() << std::endl;
  while(qry.next()) {
    ret.append(qry.value(0).toString());
  }
  return ret;
}

/*!
  Holt alle Tags, die Bildern zugeordnet wurden.

  \return Liste der Tags. Unique, aber nicht sortiert.
*/
QStringList Datenbank::getAllTags() const {
  QStringList ret;
  QSqlQuery qry(m_db);
  qry.prepare("SELECT DISTINCT tags.tag "
  	"FROM tags");
  if(!qry.exec())
    std::cerr << "Datenbankfehler: \n"
        << "Query: " << qry.lastQuery().toUtf8().constData() << '\n'
    	<< qry.lastError().databaseText().toUtf8().constData() << '\n'
	<< qry.lastError().driverText().toUtf8().constData() << std::endl;
  while(qry.next()) {
    ret.append(qry.value(0).toString());
  }
  return ret;
}

/*!
  \return Liste aller bekannten Bilder.
*/ 
QStringList Datenbank::getAllImages() const {
  QStringList ret;
  QSqlQuery qry(m_db);
  qry.prepare("SELECT bild.name "
  	"FROM bild");
  if(!qry.exec())
    std::cerr << "Datenbankfehler: \n"
        << "Query: " << qry.lastQuery().toUtf8().constData() << '\n'
    	<< qry.lastError().databaseText().toUtf8().constData() << '\n'
	<< qry.lastError().driverText().toUtf8().constData() << std::endl;
  while(qry.next()) {
    ret.append(qry.value(0).toString());
  }
  return ret;
}
