/*!
 \author Johannes Bretscher <bretscher@5sl.org>
 \date 2008-05-20
*/
/*
    Copyright (C) 2008  Johannes Bretscher

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "bildauswahl.h"
#include <QPushButton>
#include <QAction>

/*!
  Constructor
*/
Bildauswahl::Bildauswahl(QWidget *p)
	:QToolBar(tr("Bildauswahl"), p) {
  
  m_bildlist = new QComboBox(this);
  m_bildlist->setSizeAdjustPolicy(QComboBox::AdjustToContents);
  QObject::connect(m_bildlist, SIGNAL(currentIndexChanged(const QString&)),
	this, SLOT(selectedImageChanged(const QString&)));

  m_first = addAction(tr("<<"), this, SLOT(firstPressed()));
  m_prev = addAction(tr("<"), this, SLOT(prevPressed()));
  m_selectBox = addWidget(m_bildlist); m_selectBox->setVisible(true);
  m_next = addAction(tr(">"), this, SLOT(nextPressed()));
  m_last = addAction(tr(">>"), this, SLOT(lastPressed()));

  setAllowedAreas(Qt::AllToolBarAreas);
  setMovable(true);

  updateStatus();
}

/*!
  Ein neuer Stapel von Bildern wird zur Ansicht / Bearbeitung bereitgestellt

  \param bilder neue Liste der vollstaendigen Bildnamen.
*/
void Bildauswahl::setImageList(const QStringList &bilder) {
  m_bildlist->clear();
  m_bildlist->addItems(bilder);
  m_bildlist->setCurrentIndex(0);
  if(!bilder.empty())
    emit bildGeaendert(bilder.first());
  updateStatus();
}

void Bildauswahl::firstPressed() {
  if(0 != m_bildlist->currentIndex()) {
    m_bildlist->setCurrentIndex(0);
    // emit bildGeaendert(m_bildlist->currentText());
  }
  updateStatus();
}

void Bildauswahl::prevPressed() {
  if(0 != m_bildlist->currentIndex()) {
    m_bildlist->setCurrentIndex(m_bildlist->currentIndex()-1);
    // emit bildGeaendert(m_bildlist->currentText());
  }
  updateStatus();
}

void Bildauswahl::nextPressed() {
  if(m_bildlist->count()-1 > m_bildlist->currentIndex()) {
    m_bildlist->setCurrentIndex(m_bildlist->currentIndex()+1);
    // emit bildGeaendert(m_bildlist->currentText());
  }
  updateStatus();
}

void Bildauswahl::lastPressed() {
  if(m_bildlist->count()-1 > m_bildlist->currentIndex()) {
    m_bildlist->setCurrentIndex(m_bildlist->count()-1);
    // emit bildGeaendert(m_bildlist->currentText());
  }
  updateStatus();
}

void Bildauswahl::selectedImageChanged(const QString &bild) {
  emit bildGeaendert(bild);
  updateStatus();
}

/*!
  Aktiviere / Deaktiviere die Elemente der Navigationsleiste in Abhaengigkeit
  vom aktuellen Bild.
*/
void Bildauswahl::updateStatus() {
  if(0 == m_bildlist->count()) {
    m_first->setEnabled(false);
    m_prev->setEnabled(false);
    m_next->setEnabled(false);
    m_last->setEnabled(false);
  } else {
    m_first->setEnabled(true);
    m_last->setEnabled(true);

    if(0 == m_bildlist->currentIndex()) {
      m_prev->setEnabled(false);
    } else {
      m_prev->setEnabled(true);
    }
    if(m_bildlist->count() - 1 <= m_bildlist->currentIndex()) {
      m_next->setEnabled(false);
    } else {
      m_next->setEnabled(true);
    }
  }
}
