/*
    Copyright (C) 2008  Johannes Bretscher

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "mainwin.h"
#include "ui_export.h"
#include <QGraphicsScene>
#include <QFileDialog>
#include <QMessageBox>
#include <QGraphicsPixmapItem>
#include <QList>
#include <QCompleter>
#include <QDir>
#include <QSettings>
#include <QWhatsThis>
#include <iostream>

enum ExportMode {
  EM_XML,
  EM_CSV
};

/*!
  Constructor
*/
MainWin::MainWin(QWidget *p)
	: QMainWindow(p) {
  m_ui.setupUi(this);

  m_tagsuche = NULL;

  m_bild = new QGraphicsScene();
  m_ui.bild->setScene(m_bild);

  m_bildauswahl = new Bildauswahl(this);
  addToolBar(Qt::TopToolBarArea, m_bildauswahl);

  m_ui.menuHilfe->insertAction(m_ui.actionAbout, QWhatsThis::createAction(m_ui.menuHilfe));
  m_ui.menuHilfe->insertSeparator(m_ui.actionAbout);
  QSettings settings;

  QObject::connect(m_ui.actionNeu, SIGNAL(triggered()),
	this, SLOT(newPicture()));
  QObject::connect(m_ui.actionNeue_Bilder, SIGNAL(triggered()),
	this, SLOT(findNewPictures()));
  QObject::connect(m_ui.actionSpeichern, SIGNAL(triggered()),
	this, SLOT(saveData()));
  QObject::connect(m_ui.actionSuchen, SIGNAL(triggered()),
	this, SLOT(search()));
  QObject::connect(m_ui.actionExport, SIGNAL(triggered()),
	this, SLOT(exportImageList()));
  QObject::connect(m_ui.actionAbout, SIGNAL(triggered()),
	this, SLOT(about()));
  QObject::connect(m_ui.actionAboutQT, SIGNAL(triggered()),
	qApp, SLOT(aboutQt()));
  QObject::connect(m_ui.actionOptionen, SIGNAL(triggered()),
	this, SLOT(options()));
  QObject::connect(m_ui.ZoomIn, SIGNAL(clicked()),
	this, SLOT(zoomIn()));
  QObject::connect(m_ui.ZoomOut, SIGNAL(clicked()),
	this, SLOT(zoomOut()));
  QObject::connect(m_ui.ZoomOriginal, SIGNAL(clicked()),
	this, SLOT(zoomOriginal()));
  QObject::connect(m_ui.FitWidth, SIGNAL(clicked()),
	this, SLOT(fitWidth()));
  QObject::connect(m_ui.FitHeight, SIGNAL(clicked()),
	this, SLOT(fitHeight()));
  QObject::connect(m_ui.tagInput, SIGNAL(returnPressed()),
	this, SLOT(newTag()));
  QObject::connect(m_ui.tagSelect, SIGNAL(itemSelectionChanged ()),
	this, SLOT(setDirty()));
  QObject::connect(m_bildauswahl, SIGNAL(bildGeaendert(const QString&)),
	this, SLOT(displayPicture(const QString&)));

  // Can this be shortened?
  QString dbPath = settings.value("db/path", "*FALSE*").toString();
  if("*FALSE*" == dbPath) {
    QDir tmpPath(QDir::homePath()+"/.imageTagger");
    if(!tmpPath.exists()) {
      tmpPath.mkpath(tmpPath.absolutePath());
    }
    dbPath = tmpPath.absolutePath();
    settings.setValue("db/path", dbPath);
  }
  m_datenbank = new Datenbank(dbPath);
  m_ui.tagSelect->addItems(m_datenbank->getAllTags());
  m_ui.tagInput->setCompleter(new QCompleter(m_ui.tagSelect->model()));

  bool ok;
  DisplayMode dpm = static_cast<DisplayMode>(settings.value("defaultDisplayMode", "*FALSE*").toInt(&ok));
  if(ok) {
    m_defaultDisplayMode = dpm;
  } else {
    m_defaultDisplayMode = DM_ORIGINAL;
  }

  m_path = QDir::currentPath();
  m_dirty = false;
}

/*!
  \~german Lade neue Bilder zum Betrachten und taggen
  \~english load new pictures for tagging and viewing
*/
void MainWin::newPicture() {
  newPictureList(QFileDialog::getOpenFileNames(this, tr("Bild oeffnen"), m_path,
	tr("Images (*.png *.jpg *.jpeg *.JPG *.xpm *.gif)")));
}

/*!
  \~german
  Neues Bild ausgewaehlt -> Anzeigen und Tags markieren
  \param bildName Pfad zum neuen Bild
  \~english
  New picture selected -> Display and update tags
  \param bildName Path to new picture
*/
void MainWin::displayPicture(const QString &bildName) {
  if(m_dirty) {
    if(QMessageBox::Yes == QMessageBox::question(this,
    	tr("Tags wurden geaendert"),
	tr("Speichern?"),
	QMessageBox::Yes|QMessageBox::No,
	QMessageBox::Yes)) {
      saveData();
    }	
  }

  QList<QGraphicsItem *> bilder = m_bild->items();
  for(QList<QGraphicsItem *>::iterator bild = bilder.begin();
	bilder.end() != bild;
	bild++) {
    m_bild->removeItem(*bild);
  }
  
  m_bildname = bildName;

  if(!m_bildname.isEmpty()) {
    QImage showBild(m_bildname);
    if(showBild.isNull()) {
      QMessageBox::warning(this, tr("Fehler"),
	tr("Bild %1 konnte nicht geladen werden").arg(m_bildname) );
    } else {
      QRectF bildSize;
      m_bild->addItem(new QGraphicsPixmapItem(QPixmap::fromImage(showBild)));
      bildSize = m_bild->itemsBoundingRect();
      m_ui.bild->setSceneRect(bildSize);
      switch(m_defaultDisplayMode) {
        case DM_ORIGINAL:
	  zoomOriginal();
	  break;
	case DM_WIDTH:
	  fitWidth();
	  break;
	case DM_MAXWIDTH:
	  if(m_ui.bild->width() < bildSize.width()) {
	    fitWidth();
	  } else {
	    zoomOriginal();
	  }
	  break;
	case DM_HEIGHT:
	  fitHeight();
	  break;
	case DM_MAXHEIGHT:
	  if(m_ui.bild->height() < bildSize.height()) {
	    fitHeight();
	  } else {
	    zoomOriginal();
	  }
	  break;
      }
    }
  } else {
    m_ui.bild->setSceneRect(m_bild->itemsBoundingRect());
  }

  statusBar()->showMessage(m_bildname);

  m_path = m_bildname.left(m_bildname.lastIndexOf('/'));

  // Tagdisplay aktualisieren
  QList<QListWidgetItem*> selectedTags = m_ui.tagSelect->selectedItems();
  QList<QListWidgetItem*>::iterator itTag;
  for(itTag = selectedTags.begin();
  	selectedTags.end() != itTag;
	itTag++) {
    (*itTag)->setSelected(false);
  }
  QStringList taglist = m_datenbank->getTags(m_bildname);
  QStringList::iterator itNewtag;
  for(itNewtag=taglist.begin();
	taglist.end() != itNewtag;
	itNewtag++) {
    m_ui.tagSelect->findItems(*itNewtag, Qt::MatchExactly).first()->setSelected(true);
  }
  m_dirty = false;
}

/*!
  Sucht in einem Verzeichnis nach Bildern, die noch nicht in der Datenbank sind.
*/ 
void MainWin::findNewPictures() {
  QStringList newImages;

  QString pathName = QFileDialog::getExistingDirectory(this, tr("Bilderpfad"), m_path);
  if("" != pathName) {
    QDir path(pathName);
    QStringList bildtyp;
    bildtyp << "*.png" << "*.jpg" <<  "*.jpeg" << "*.xpm" <<  "*.gif";
    QStringList dirBilder = path.entryList(bildtyp, QDir::Files|QDir::Readable, QDir::Name);
    if(!dirBilder.empty()) {
      QStringList alleBilder = m_datenbank->getAllImages();
      alleBilder.sort();

      int i=0, j=0;
      while(i<dirBilder.size() && j < alleBilder.size()) {
        if(!dirBilder[i].startsWith(pathName)) 
          dirBilder[i] = pathName + (pathName.endsWith('/')?"":"/") + dirBilder[i];
        if(dirBilder[i] < alleBilder[j]) {
	  newImages << dirBilder[i++];
	} else if(dirBilder[i] > alleBilder[j]) {
	  j++;
	} else {
	  i++;
	  j++;
	}
      }
      while(i<dirBilder.size()) { // Noch Bilder uebrig?
        if(!dirBilder[i].startsWith(pathName)) 
          dirBilder[i] = pathName + (pathName.endsWith('/')?"":"/") + dirBilder[i];

        newImages << dirBilder[i++];
      }
    }
  }

  if(newImages.empty()) {
    QMessageBox::information(this, tr("Suche"), tr("keine neuen Bilder gefunden"));
  } else {
    newPictureList(newImages);
  }

}

/*!
  \~german
  Tags zum Bild in die Datenbank schreiben
  \~english
  Write tags into database
*/
void MainWin::saveData() {
  QStringList taglist;
  QList<QListWidgetItem*> selectedTags = m_ui.tagSelect->selectedItems();
  QList<QListWidgetItem*>::iterator itTag;
  for(itTag = selectedTags.begin();
  	selectedTags.end() != itTag;
	itTag++) {
    taglist.append((*itTag)->text());
  }
  m_datenbank->writeTags(m_bildname, taglist);
  m_dirty = false;
}

/*!
  About Informationen anzeigen
*/ 
void MainWin::about() {
  QMessageBox::about(this, tr("Ueber imageTagger"),
    tr("imageTagger ist ein einfaches Programm zum organisieren von "
       "Bildersammlungen ueber Tags."
       "<p>"
       "&copy;2008 <a href=\"mailto:bretscher@5sl.org\">Johannes Bretscher</a>"
       "<p>"
       "This program is distributed in the hope that it will be useful, "
    "but WITHOUT ANY WARRANTY; without even the implied warranty of "
    "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the "
    "GNU General Public License for more details. "));
}

/*!
  \~german
  Bildanzeige vergroessern
  \~english
  Increase picture display
*/
void MainWin::zoomIn() {
  m_ui.bild->scale(2.0, 2.0);
}

/*!
  \~german
  Bildanzeige verkleinern
  \~english
  Decrease picture display
*/
void MainWin::zoomOut() {
  m_ui.bild->scale(0.5, 0.5);
}

/*!
 Setzt die Matrix auf Einheitsmatrix zurueck
*/ 
void MainWin::zoomOriginal() {
  m_ui.bild->resetMatrix();
}

/*!
  Passt die Bildbreite an die Fensterbreite an.
*/
void MainWin::fitWidth() {
  float zoom =  m_ui.bild->width() / m_ui.bild->sceneRect().width();

  m_ui.bild->setMatrix(QMatrix(zoom, 0.0, 0.0, zoom, 0.0, 0.0), false);
}

/*!
  Passt die Bildhoehe an die Fensterhoehe an.
*/
void MainWin::fitHeight() {
  float zoom =  m_ui.bild->height() / m_ui.bild->sceneRect().height();

  m_ui.bild->setMatrix(QMatrix(zoom, 0.0, 0.0, zoom, 0.0, 0.0), false);
}

/*!
  \~german
  Tag wurde in der Eingabezeile eingegeben.
  \~english
  New tags entered
*/
void MainWin::newTag() {
  QString tagText = m_ui.tagInput->text();
  if("" == tagText)
    return;	// Ignore empty tags

  QList<QListWidgetItem*> tagList = m_ui.tagSelect->findItems(tagText, Qt::MatchExactly);
  if(!tagList.empty()) {
    tagList.first()->setSelected(true);
  } else {
    QListWidgetItem *newTagItem = new QListWidgetItem(tagText, m_ui.tagSelect);
    newTagItem->setSelected(true);
  }
  m_ui.tagInput->clear();

  setDirty();
}

/*!
  \~german
  Tags wurden geaendert
  \~english
  Tags have been changed
*/
void MainWin::setDirty() {
  m_dirty = true;
}

/*!
  \~german Suche Bilder nach Tags
  \~english Search pictures with tags
*/
void MainWin::search() {
  m_tagsuche = new Tagsuche(m_datenbank, this);
  QObject::connect(m_tagsuche, SIGNAL(selected(const QStringList &)),
	this, SLOT(newPictureList(const QStringList &)));
  m_tagsuche->show();
}

void MainWin::options() {
  Optionen optionen(this);
  QObject::connect(&optionen, SIGNAL(setDefaultZoom(DisplayMode)),
	this, SLOT(changeDefaultZoom(DisplayMode)));
  optionen.exec();

}
/*!
  \~german Durch Laden oder Bildsuche hat sich die Liste der Bilder geaendert.
  \~english Image list has changed
*/
void MainWin::newPictureList(const QStringList &list) {
  // std::cerr << "Piclist" << std::endl;
  m_pictureList = list;
  m_bildauswahl->setImageList(list);
}

/*!
  Aendert den voreingestellten Zoom fuer neu geladene Bilder.

  \param newDM ID fuer neuen Zoommodus
*/ 
void MainWin::changeDefaultZoom(DisplayMode newDM) {

  if(newDM != m_defaultDisplayMode) {
    QSettings settings;
    settings.setValue("defaultDisplayMode", newDM);
    m_defaultDisplayMode = newDM;
  }
}

void MainWin::exportImageList() {
  QDialog exportWin(this);
  Ui::Export ui;

  ui.setupUi(&exportWin);
  ui.format->addItem("CSV", EM_CSV);
  ui.format->addItem("XML", EM_XML);

  if(QDialog::Accepted == exportWin.exec()) {
    QString filename = QFileDialog::getSaveFileName(this, tr("Export"), m_path);
    
    if(0 != filename.size()) {
      QFile file(filename, this);
      if(file.open(QIODevice::WriteOnly|QIODevice::Text)) {
        if(EM_XML == ui.format->itemData(ui.format->currentIndex()).toUInt()) {
	  file.write("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n<bilder>");
	}
	QStringList taglist;
	for(int i=0; i<m_pictureList.size();i++) {
	  if(ui.addTags->isChecked()) {
	    taglist = m_datenbank->getTags(m_pictureList[i]);
	  }
	  switch(static_cast<ExportMode>(ui.format->itemData(ui.format->currentIndex()).toUInt())) {
	    case EM_CSV:
	      file.write("\"");
	      file.write(m_pictureList[i].toUtf8());
	      file.write("\"");
	      for(int j=0; j<taglist.size(); j++) {
	        file.write(";\"");
		file.write(taglist[j].toUtf8());
		file.write("\"");
	      }
	      file.write("\n");
	      break;
	    case EM_XML:
	      file.write(QString("<bild><name>%1</name>")
	      	.arg(m_pictureList[i])
		.toUtf8());
	      for(int j=0; j<taglist.size(); j++) {
	        file.write(QString("<tag>%1</tag>")
			.arg(taglist[j])
			.toUtf8());
	      }
	      file.write("</bild>");
	      break;
	  }
	}
        if(EM_XML == ui.format->itemData(ui.format->currentIndex()).toUInt()) {
	  file.write("</bilder>");
	}
	file.close();
      } else {
        QMessageBox::warning(this, tr("Datei"), tr("Kann Datei %1 nicht oeffnen: %2")
		.arg(filename)
		.arg(file.error()));
      }
    }
  }
}
